// Copyright 2005, 2006, 2007 Dennis van Weeren
//
// This file is part of Minimig
//
// Minimig is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 3 of the License, or
// (at your option) any later version.
//
// Minimig is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
//
//
// S19 assembler output file to verilog rom table converter
// Written by Dennis van Weeren for the Minimig project
//
// ??-??-2005	-first version
// 27-11-2006	-adapted to code to generate tables for use with blockram
// 04-07-2007	-cleaned up code for open-source release

#include <stdio.h>
#include <stdlib.h>

#define     LEN 			256				/*size of working buffer*/
#define			ROMSIZE		1024			/*size rom in words*/
#define			VERBOSE		0					/*verbose on/off*/

/*functions*/
unsigned char GetByte(unsigned char *s);

/*global variables*/
unsigned char buf[LEN];
unsigned short rom[ROMSIZE];

/*code starts here*/
int main(int argc, char *argv[])
{
  FILE *in,*out;
  int n;
  unsigned long x,address,code;
  unsigned char *p;
  
  /*check for arguments*/
  if(argc<3)
  {
		printf("not enough arguments!\n");
		printf("usage: s2vrlg <input.s19> <output.v>\n");
		return(0);
	}
	else if(argc>3)
	{
		printf("too many arguments!\n");
		printf("usage: s2vrlg <input.s19> <output.v>\n");
		return(0);
	}
   
  /*open input S19 file*/
  in=fopen(argv[1],"r");
  if(!in)
  {
		printf("could not open %s\n",argv[1]);
		return(0);
	}
	/*read input S19 file and convert to rom image*/
	while(1)
  {
    /*read one record from the file*/
    n=fscanf(in,"%s",buf);

    /*exit if end of file*/
    if(n==EOF)
    {
			#if VERBOSE==1
      	printf("end of file\n");
			#endif
      break;
    }

    /*check for s record*/
    if(buf[0]!='S')
    {
      printf("not an s19 record\n");
      break;
    }

    /*get number of bytes in file and pointer to next byte pair*/
    x=GetByte(&buf[2]);
    p=&buf[4];
    
		#if VERBOSE==1
			printf("length of S19 line->%d, ",x);
		#endif
		
    /*examine type (get address)*/
    address=0;
    switch(buf[1])
    {
          case '3':/*4 byte adress*/
            address+=(unsigned long)GetByte(p)*16777216L;
            p+=2;
            x--;
          case '2':/*3 byte adress*/
            address+=(unsigned long)GetByte(p)*65536L;
            p+=2;
            x--;
          case '1':/*2 byte adress*/
            address+=(unsigned long)GetByte(p)*256L;
            p+=2;
            x--;
            address+=(unsigned long)GetByte(p);
            p+=2;
            x--;
            
						#if VERBOSE==1
							printf("valid s-record entry\n");
            #endif
            
						/*now get data from rest of line*/
            while(x>=2)
            {
              /*get one word of data*/
              code=GetByte(p)*256;
              p+=2;
              code+=GetByte(p);
              p+=2;
              x-=2;
              
              /*put rom word into rom*/
              if(address/2>=ROMSIZE)
              	printf("address out of range! rom not valid!->%04ld\n",address);
              else
              	rom[address/2]=code;
                              
							/*next address*/
              address+=2;
            }
            
            break;

          default:
            printf("unknown s-record entry->%c\n",buf[1]);
            break;
    }
  }
	fclose(in);

	/*open output .V file*/	
	out=fopen(argv[2],"w");
 	if(!out)
	{
		printf("could not open %s\n",argv[2]);
		return(0);
	}  
	/*write rom image to output file*/
	for(n=0;n<ROMSIZE;n++)
		fprintf(out,"\t\t%04ld:\tromdata[15:0]=16'h%04X;\n",n,rom[n]);
	/*close files*/
	fclose(out);
  
  printf("hit a key to exit\n");
  scanf("%c",buf);
  
  return 0;
}

/*Get a single binary byte from the S19 line by combining 2 hex ascii values*/
unsigned char GetByte(unsigned char *s)
{
    int x;
    sscanf(s,"%2x",&x);
    return((unsigned char) x);
}
